#include <stdio.h>
#include <sys/types.h>
#include <pthread.h>
#include <unistd.h>
#include <stdlib.h>
#include "funadevice.h"

//define
#define NUM_OF_PORTS 3
#define NUM_OF_DEVICES_IN_EACH_PORT 8
#define NUM_OF_MAX_REVERSING_TIMES 1024

//global variables
myDevice_t usbDev;
int isDebug = 0; //don't delete it
unsigned char valveA_state;
unsigned char valveB_state;
unsigned char valveC_state;
double controlling_time;

pthread_mutex_t mutexA = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mutexB = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mutexC = PTHREAD_MUTEX_INITIALIZER;

//structure
typedef struct _param{
  int valve_order; //0 ~ num_of_device_in_each_port-1
  double reversing_times[NUM_OF_MAX_REVERSING_TIMES]; //times at which valve state is reversed
  int num_of_reversing_times;
  char port_id; //A or B or C
}param;

//function for transform decimal to binary string
char* decimal_to_8bit_binary(unsigned char dec, char* str){
  int i;
  str[8] = '\0';
  for(i = 0; i < 8; i++){
    if(dec & (1 << i)){
      str[7-i] = '1';
    }else{
      str[7-i] = '0';
    }
  }
  return str;
}

//function for procesing valve on/off pattern
void* pattern(void* arg)
{
  param *th_param = (param*)arg;
  char str[9];
  int reverse_count = 0;
  
  while(reverse_count != th_param->num_of_reversing_times){
    //sleep
    if(reverse_count != 0){
      usleep((long)((th_param->reversing_times[reverse_count]-th_param->reversing_times[reverse_count-1])*1e6));
    }else{
      usleep((long)(th_param->reversing_times[reverse_count]*1e6));
    }
    //valve state change
    switch(th_param->port_id){
    case 'A':
      pthread_mutex_lock(&mutexA);
      if(valveA_state & (1 << th_param->valve_order)){
	valveA_state -= 1 << th_param->valve_order;
      }else{
	valveA_state += 1 << th_param->valve_order;
      }
      writePortA(&usbDev, valveA_state);
      printf("Valve %d of PortA reversed at %lf. valve state is %s.\n", th_param->valve_order+1, th_param->reversing_times[reverse_count], decimal_to_8bit_binary(valveA_state, str));
      pthread_mutex_unlock(&mutexA);
      break;
    case 'B':
      pthread_mutex_lock(&mutexB);
      if(valveB_state & (1 << th_param->valve_order)){
	valveB_state -= 1 << th_param->valve_order;
      }else{
	valveB_state += 1 << th_param->valve_order;
      }
      writePortB(&usbDev, valveB_state);
      printf("Valve %d of PortB reversed at %lf. valve state is %s.\n", th_param->valve_order+1, th_param->reversing_times[reverse_count], decimal_to_8bit_binary(valveB_state, str));
      pthread_mutex_unlock(&mutexB);
      break;
    case 'C':
      pthread_mutex_lock(&mutexC);
      if(valveC_state & (1 << th_param->valve_order)){
	valveC_state -= 1 << th_param->valve_order;
      }else{
	valveC_state += 1 << th_param->valve_order;
      }
      writePortC(&usbDev, valveC_state);
      printf("Valve %d of PortC reversed at %lf. valve state is %s.\n", th_param->valve_order+1, th_param->reversing_times[reverse_count], decimal_to_8bit_binary(valveC_state, str));
      pthread_mutex_unlock(&mutexC);
      break;
    default:
      break;
    }   
    reverse_count++;
  }

  return NULL;
}

//main function
int main()
{
  int i;
  pthread_t thread_id[NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT];
  int status;
  void *result;
  char str[9];

  //create valve_param
  param **valve_param;
  valve_param = (param**)malloc(sizeof(param*)*NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT);
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    valve_param[i] = (param*)malloc(sizeof(param));
  }

  //assign port_id
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    if(i < NUM_OF_DEVICES_IN_EACH_PORT){
      valve_param[i]->port_id = 'A';
    }else if(i < 2*NUM_OF_DEVICES_IN_EACH_PORT){
      valve_param[i]->port_id = 'B';
    }else{
      valve_param[i]->port_id = 'C';
    }
  }

  //assign valve_order
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    if(i < NUM_OF_DEVICES_IN_EACH_PORT){
      valve_param[i]->valve_order = i;
    }else if(i < 2*NUM_OF_DEVICES_IN_EACH_PORT){
      valve_param[i]->valve_order = i-8;
    }else{
      valve_param[i]->valve_order = i-16;
    }
  }

  //open device
  openDevice(&usbDev);
  identifyDevice(&usbDev);
  setOutputMode(&usbDev);

  //assign controlling time
  #include "controlling_time.config"

  //assign valve initial state
  //include valve initial state description file
  #include "valve_initial_state.config"
  writePortA(&usbDev, valveA_state);
  writePortB(&usbDev, valveB_state);
  writePortC(&usbDev, valveC_state);
  printf("Valve initial state of PortA:%s\n", decimal_to_8bit_binary(valveA_state, str));
  printf("Valve initial state of PortB:%s\n", decimal_to_8bit_binary(valveB_state, str));
  printf("Valve initial state of PortC:%s\n", decimal_to_8bit_binary(valveC_state, str));

  //assign valve state reversing time
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    valve_param[i]->num_of_reversing_times = 0;
  }
  //include valve reversing pattern description file
  #include "valve_reversing_pattern.config"

  //calculate last reversing time
  double last_reversing_time = 0;
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    if(valve_param[i]->num_of_reversing_times != 0 && valve_param[i]->reversing_times[valve_param[i]->num_of_reversing_times-1] > last_reversing_time){
      last_reversing_time = valve_param[i]->reversing_times[valve_param[i]->num_of_reversing_times-1];
    }
  }

  //start controlling
  printf("start controlling!\n");
  //parallel pthread processing
  printf("start multi processing!\n");
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    status = pthread_create(&thread_id[i], NULL, pattern, (void*)valve_param[i]);
    if(status!=0){
      fprintf(stderr, "pthread_create : %s", strerror(status));
    }
  }
  //clean up pthread
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    pthread_join(thread_id[i], &result);
  }
  printf("finish multi processing!\n");
  //sleep untill the end of controlling time
  usleep((long)((controlling_time-last_reversing_time)*1e6));
  //finish controlling
  printf("finish controlling!\n");

  //switch off all valve
  valveA_state = 0;
  valveB_state = 0;
  valveC_state = 0;
  writePortA(&usbDev, valveA_state);
  writePortB(&usbDev, valveB_state);
  writePortC(&usbDev, valveC_state);

  //close device
  closeDevice(&usbDev);

  //free valve_param
  for(i = 0; i < NUM_OF_PORTS*NUM_OF_DEVICES_IN_EACH_PORT; i++){
    free(valve_param[i]);
  }
  free(valve_param);

  return 0;
}
